<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
// استبدال DomPDF بـ mPDF
// use Barryvdh\DomPDF\Facade\Pdf;
use Mpdf\Mpdf;
use Mpdf\Output\Destination;
use Mpdf\Config\ConfigVariables;
use Mpdf\Config\FontVariables;
use Prgayman\Zatca\Facades\Zatca;
use Prgayman\Zatca\Utilis\QrCodeOptions;

class PdfController extends Controller
{
    /**
     * Download invoice as PDF using Blade view.
     */
    public function generate(Invoice $invoice)
    {
        // Ensure items are loaded and totals are consistent
        $invoice->load('items');

        $calculatedSubtotal = (float) $invoice->items->sum('line_total');
        if ($calculatedSubtotal > 0 && ((float)$invoice->tax === 0.0 || (float)$invoice->subtotal !== $calculatedSubtotal)) {
            $invoice->subtotal = $calculatedSubtotal;
            $invoice->tax = round($calculatedSubtotal * 0.15, 2);
            $invoice->total = round($invoice->subtotal + $invoice->tax, 2);
            $invoice->save();
        }

        // إعدادات الشركة وQR للفاتورة
        $setting = \App\Models\Setting::query()->first();
        $logoDataUri = null;
        $qrDataUri = null;

        // تحويل شعار الشركة إلى data URI من التخزين الفعلي (وليس رابط عام)
        if (!empty($setting?->company_logo_path)) {
            $logoPath = storage_path('app/public/' . $setting->company_logo_path);
            if (file_exists($logoPath)) {
                try {
                    $logoData = file_get_contents($logoPath);
                    $mime = function_exists('mime_content_type') ? mime_content_type($logoPath) : null;
                    if (!$mime) {
                        $ext = strtolower(pathinfo($logoPath, PATHINFO_EXTENSION));
                        $mime = $ext === 'png' ? 'image/png' : ($ext === 'webp' ? 'image/webp' : 'image/jpeg');
                    }
                    $logoDataUri = 'data:' . $mime . ';base64,' . base64_encode($logoData);
                } catch (\Throwable $e) {
                    $logoDataUri = null;
                }
            }
        }

        // توليد رمز ZATCA كصورة PNG وتمريره كـ data URI
        if (!empty($setting?->company_name) && !empty($setting?->company_tax_number)) {
            $vatNumber = preg_replace('/\D+/', '', (string)($setting->company_tax_number));
            if (strlen($vatNumber) === 15) {
                $timestamp = ($invoice->date ?? $invoice->created_at ?? now())->setTimezone('UTC')->format('Y-m-d\\TH:i:s\\Z');
                $totalWithVat = number_format((float)($invoice->total ?? 0), 2, '.', '');
                $vatTotal = number_format((float)($invoice->tax ?? 0), 2, '.', '');
                try {
                    $options = new QrCodeOptions();
                    $options->format('png');
                    $options->size(320);
                    $options->margin(0);
                    $qrBinary = Zatca::sellerName($setting->company_name)
                        ->vatRegistrationNumber($vatNumber)
                        ->timestamp($timestamp)
                        ->totalWithVat($totalWithVat)
                        ->vatTotal($vatTotal)
                        ->toQrCode($options);
                    $qrDataUri = 'data:image/png;base64,' . base64_encode($qrBinary);
                } catch (\Throwable $e) {
                    $qrDataUri = null;
                }
            } else {
                $qrDataUri = null;
            }
        }

        // توليد HTML من القالب ثم تحويله عبر mPDF
        $html = view('pdf.invoice', [
            'invoice' => $invoice,
            'setting' => $setting,
            'qrDataUri' => $qrDataUri,
            'logoDataUri' => $logoDataUri,
        ])->render();

        // إعداد مجلد مؤقت للمكتبة لتفادي مشاكل صلاحيات ويندوز
        $tempDir = storage_path('app/mpdf-temp');
        if (!is_dir($tempDir)) {
            @mkdir($tempDir, 0777, true);
        }

        $mpdfConfig = [
            'mode' => 'utf-8',
            'format' => 'A4',
            'orientation' => 'P',
            'tempDir' => $tempDir,
            'autoScriptToLang' => true,
            'autoLangToFont' => true,
        ];

        // ضبط خط Alexandria إن كان موجودًا داخل public/fonts
        $fontDir = public_path('fonts');
        $alexR = $fontDir . DIRECTORY_SEPARATOR . 'Alexandria-Regular.ttf';
        $alexB = $fontDir . DIRECTORY_SEPARATOR . 'Alexandria-Bold.ttf';
        if (file_exists($alexR) && file_exists($alexB)) {
            $defaultConfig = (new ConfigVariables())->getDefaults();
            $fontDirs = $defaultConfig['fontDir'];
            $defaultFontConfig = (new FontVariables())->getDefaults();
            $fontData = $defaultFontConfig['fontdata'];
            $mpdfConfig['fontDir'] = array_merge($fontDirs, [$fontDir]);
            $mpdfConfig['fontdata'] = $fontData + [
                'alexandria' => [
                    'R' => 'Alexandria-Regular.ttf',
                    'B' => 'Alexandria-Bold.ttf',
                ],
            ];
            $mpdfConfig['default_font'] = 'alexandria';
        }

        $mpdf = new Mpdf($mpdfConfig);
        $mpdf->WriteHTML($html);

        // استخدام الرمز 'S' لضمان إرجاع المحتوى كسلسلة نصية متوافقة مع جميع إصدارات mPDF
        $content = $mpdf->Output('', 'S');
        return response($content, 200)
            ->header('Content-Type', 'application/pdf')
            ->header('Content-Disposition', 'attachment; filename="invoice_' . $invoice->id . '.pdf"');
    }
}