<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Quote;
use App\Models\QuoteItem;
use App\Models\Setting;
// إضافة استخدام مكتبة ZATCA
use Prgayman\Zatca\Facades\Zatca;
use Prgayman\Zatca\Utilis\QrCodeOptions;

class QuoteController extends Controller
{
    public function index(Request $request)
    {
        $query = Quote::query();
        $totalCount = $query->count();
        $quotes = $query->orderByDesc('id')->get()->map(function (Quote $q) {
            return [
                'id' => $q->id,
                'number' => $q->number,
                'customer_name' => $q->customer_name,
                'date' => $q->date ? $q->date->toDateString() : '',
                'total' => (float) $q->total,
                'status' => $q->status,
            ];
        });

        return view('quotes.index', [
            'quotes' => $quotes,
            'totalCount' => $totalCount,
        ]);
    }

    public function create()
    {
        return view('quotes.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'number' => ['required', 'string', 'max:50'],
            'customer_name' => ['required', 'string', 'max:255'],
            'customer_phone' => ['required', 'string', 'max:50'],
            'customer_address' => ['nullable', 'string', 'max:255'],
            'customer_tax_number' => ['nullable', 'string', 'max:50'],
            'date' => ['required', 'date'],
            'status' => ['nullable', 'in:مسودة,موافق عليه,مرفوض'],
            'notes' => ['nullable', 'string', 'max:1000'],
            'items' => ['required', 'array', 'min:1'],
            'items.*.description' => ['required', 'string', 'max:255'],
            'items.*.quantity' => ['required', 'numeric', 'min:1'],
            'items.*.unit_price' => ['required', 'numeric', 'min:0'],
        ]);

        $quote = null;
        DB::transaction(function () use (&$quote, $validated) {
            $items = $validated['items'];
            $subtotal = 0.0;
            foreach ($items as $it) {
                $subtotal += ((int)$it['quantity']) * ((float)$it['unit_price']);
            }

            $quote = new Quote();
            $quote->number = $validated['number'];
            $quote->customer_name = $validated['customer_name'];
            $quote->customer_phone = $validated['customer_phone'];
            $quote->customer_address = $validated['customer_address'] ?? null;
            $quote->customer_tax_number = $validated['customer_tax_number'] ?? null;
            $quote->date = $validated['date'];
            $quote->status = $validated['status'] ?? 'مسودة';
            $quote->notes = $validated['notes'] ?? null;
            $quote->subtotal = $subtotal;
            $quote->tax = round($subtotal * 0.15, 2);
            $quote->total = round($quote->subtotal + $quote->tax, 2);
            $quote->save();

            foreach ($items as $it) {
                $qty = (int) $it['quantity'];
                $price = (float) $it['unit_price'];
                $lineTotal = $qty * $price;

                $quote->items()->create([
                    'description' => $it['description'],
                    'quantity' => $qty,
                    'unit_price' => $price,
                    'line_total' => $lineTotal,
                ]);
            }
        });

        return redirect()->route('quotes.show', $quote)->with('success', 'تم إنشاء عرض السعر بنجاح.');
    }

    public function show(Quote $quote)
    {
        $quote->load('items');

        $calculatedSubtotal = (float) $quote->items->sum('line_total');
        if ($calculatedSubtotal > 0 && ((float)$quote->tax === 0.0 || (float)$quote->subtotal !== $calculatedSubtotal)) {
            $quote->subtotal = $calculatedSubtotal;
            $quote->tax = round($calculatedSubtotal * 0.15, 2);
            $quote->total = round($quote->subtotal + $quote->tax, 2);
            $quote->save();
        }

        // إعدادات الشركة وتوليد رمز ZATCA بنفس أسلوب الفواتير
        $setting = Setting::first();
        $qrCodeSvg = null;
        if (!empty($setting?->company_name) && !empty($setting?->company_tax_number)) {
            $timestamp = ($quote->date ?? $quote->created_at ?? now())->setTimezone('UTC')->format('Y-m-d\\TH:i:s\\Z');
            $totalWithVat = number_format((float)($quote->total ?? 0), 2, '.', '');
            $vatTotal = number_format((float)($quote->tax ?? 0), 2, '.', '');
            try {
                $options = new QrCodeOptions();
                $options->format('svg');
                $options->size(140);
                $options->margin(0);
                $qrCodeSvg = Zatca::sellerName($setting->company_name)
                    ->vatRegistrationNumber($setting->company_tax_number)
                    ->timestamp($timestamp)
                    ->totalWithVat($totalWithVat)
                    ->vatTotal($vatTotal)
                    ->toQrCode($options);
            } catch (\Throwable $e) {
                $qrCodeSvg = null;
            }
        }

        return view('quotes.show', [
            'quote' => $quote,
            'setting' => $setting,
            'qrCodeSvg' => $qrCodeSvg,
        ]);
    }

    public function edit(Quote $quote)
    {
        $quote->load('items');
        return view('quotes.edit', ['quote' => $quote]);
    }

    public function update(Request $request, Quote $quote)
    {
        $validated = $request->validate([
            'number' => ['required', 'string', 'max:50'],
            'customer_name' => ['required', 'string', 'max:255'],
            'customer_phone' => ['required', 'string', 'max:50'],
            'customer_address' => ['nullable', 'string', 'max:255'],
            'customer_tax_number' => ['nullable', 'string', 'max:50'],
            'date' => ['nullable', 'date'],
            'status' => ['nullable', 'in:مسودة,موافق عليه,مرفوض'],
            'notes' => ['nullable', 'string', 'max:1000'],
            'items' => ['required', 'array', 'min:1'],
            'items.*.description' => ['required', 'string', 'max:255'],
            'items.*.quantity' => ['required', 'numeric', 'min:1'],
            'items.*.unit_price' => ['required', 'numeric', 'min:0'],
        ]);

        DB::transaction(function () use ($quote, $validated) {
            $items = $validated['items'];
            $subtotal = 0.0;
            foreach ($items as $it) {
                $subtotal += ((int)$it['quantity']) * ((float)$it['unit_price']);
            }

            $quote->number = $validated['number'];
            $quote->customer_name = $validated['customer_name'];
            $quote->customer_phone = $validated['customer_phone'];
            $quote->customer_address = $validated['customer_address'] ?? null;
            $quote->customer_tax_number = $validated['customer_tax_number'] ?? null;
            if (isset($validated['date'])) {
                $quote->date = $validated['date'];
            }
            if (isset($validated['status'])) {
                $quote->status = $validated['status'];
            }
            $quote->notes = $validated['notes'] ?? null;
            $quote->subtotal = $subtotal;
            $quote->tax = round($subtotal * 0.15, 2);
            $quote->total = round($quote->subtotal + $quote->tax, 2);
            $quote->save();

            // استبدال البنود: حذف القديمة وإنشاء الجديدة
            $quote->items()->delete();
            foreach ($items as $it) {
                $qty = (int) $it['quantity'];
                $price = (float) $it['unit_price'];
                $lineTotal = $qty * $price;

                $quote->items()->create([
                    'description' => $it['description'],
                    'quantity' => $qty,
                    'unit_price' => $price,
                    'line_total' => $lineTotal,
                ]);
            }
        });

        return redirect()->route('quotes.show', $quote)->with('success', 'تم تحديث عرض السعر بنجاح.');
    }

    public function destroy(Quote $quote)
    {
        DB::transaction(function () use ($quote) {
            $quote->items()->delete();
            $quote->delete();
        });
        return redirect()->route('quotes.index')->with('success', 'تم حذف عرض السعر بنجاح.');
    }
}